import serial
import json
import time
import sys

class HummingbirdError(Exception):
    """Exception raised for errors in the Hummingbird device."""
    pass

class Hummingbird:
    """Interface for the Hummingbird laser device."""

    def __init__(self, port, timeout=1):
        """
        Initialize the Hummingbird device connection.

        Args:
            port (str): Serial port (e.g., '/dev/ttyUSB0' or 'COM3').
            timeout (int): Serial timeout in seconds.
        """
        self.serial = serial.Serial(port, 115200, timeout=timeout)

    def _send_command(self, command):
        """
        Send a command and return the response.

        Args:
            command (str): Command string without newline.

        Returns:
            tuple: data (str/dict/None)
        """
        self.serial.write(f"{command}\n".encode('ascii'))

        response = self.serial.readline().decode('ascii').strip()
        if not response:
            raise HummingbirdError("No response from device")

        if response.startswith("OK"):
            data_part = response[3:].strip()
            if not data_part:
                return None

            # Try parsing as JSON if it looks like it
            try:
                return json.loads(data_part)
            except json.JSONDecodeError:
                return data_part

        elif response.startswith("FAIL"):
            error_desc = response[5:].strip()
            raise HummingbirdError(error_desc)
        else:
            raise HummingbirdError(f"Unexpected response: {response}")

    def on(self) -> None:
        """Turn the laser on."""
        self._send_command("on")

    def off(self) -> None:
        """Turn the laser off."""
        self._send_command("off")

    def status(self) -> dict:
        """Retrieve the status of the laser."""
        return self._send_command("status?")

    def clear(self) -> None:
        """Clear the error state."""
        self._send_command("clear")

    def is_on(self) -> bool:
        """Check if the laser is on."""
        return self._send_command("on?")

    def is_off(self) -> bool:
        """Check if the laser is off."""
        return self._send_command("off?")

    def close(self) -> None:
        """Close the serial connection."""
        if self.serial and self.serial.is_open:
            self.serial.close()

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.close()

if __name__ == "__main__":
    if len(sys.argv) != 3:
        print("Usage: python hummingbird.py <port> <command>")
        sys.exit(1)

    port = sys.argv[1]
    command = sys.argv[2]

    try:
        with Hummingbird(port) as hb:
            result = hb._send_command(command)
            if result is None:
                print("OK")
            elif isinstance(result, (dict, list)):
                print(json.dumps(result, indent=2))
            else:
                print(result)
    except HummingbirdError as e:
        print(f"FAIL {e}")
        sys.exit(1)
    except Exception as e:
        print(f"Error: {e}")
        sys.exit(1)
